/* App State */
class AppState {
	constructor() {
		this.version = null;
		try {
			const manifest = chrome.runtime.getManifest();
			this.version = manifest?.version;
		} catch (e) {
			console.error(e);
		}
		this.state = { // persisted state
			settings: {}, // Extension specific settings, User settings are managed in web app.
			thread_sort_mode: 'date_desc', // default to newest sort mode
		};

		// settings
		this.settingsDefault = {
			server_url: "https://solfray.com", // fallback server url
			server_api_key: '',
			font_size: '0.7em',
		};
		this.settingsSchema = {
			server_url: 'string',
			server_api_key: 'string',
			font_size: 'string',
		};
		this.settingsDescriptions = {
			server_url: 'The URL of the server to send chats to.',
			server_api_key: 'Your API key for interfacing with the server.',
			font_size: 'General font size of extension text. Some elements may be larger or smaller.',
		};
		for (let key in this.settingsDefault) this.settingsSchema[key] = typeof this.settingsDefault[key];
		this.settingsSchema.server_url = 'string';

		// other session state variables (not persisted)
		this.lockedThreadId = null; // set to id if locked
		this.currentURL = null; // The current URL being viewed by the user.
		this.loadState();

		// Uses class CUN2 from thread.js
		this.threadApp = null; // set externally after thread.js is loaded
	}

	getCurrentURL() {
		return this.currentURL || null;
	}

	setCurrentURL(url) {
		this.currentURL = null;
		$('#thread-container').empty();
		if (typeof url == 'string' && url.length > 0 && url.toLowerCase().startsWith('https://')) {
			this.currentURL = url;
			$('#thread-container').append(`<div class="col-md-8 offset-md-2"><span class="text-muted small">${this.currentURL}</span></div>`)
		}
		return this.currentURL;
	}

	feed(arg, err = false) {
		if (err) console.trace(arg);	// for debugging
		$('#feed_error').toggle((err ? true : false));
		$('#feed').empty().append((arg.toString() || "&nbsp;"));
	}

	lockThread() {
		// Will prevent re-render on user navigation in browser.
		this.lockedThreadId = (this.threadApp && typeof this.threadApp == 'object')? (this.threadApp?.thread_id || null): null;
		return this.lockedThreadId ? true : false;
	}

	unlockThread() {
		// Will allow re-render on user navigation in browser.
		this.lockedThreadId = null;
		return false;
	}

	toggleThreadLock() {
		if (this.lockedThreadId) return this.unlockThread();
		return this.lockThread();
	}

	getShortURL() {
		const url_len = 30;
		const url = this.currentURL || null;
		if (!url || typeof url !== 'string') return null;
		var shortUrl = url.substring(0, url_len);
		return url.length > url_len ? shortUrl + "..." : url + "";
	}

	getThreads(url = null) {
		if (this.lockedThreadId) return;
		if (url) this.setCurrentURL(url); // will ignore non-https urls
		url = this.getCurrentURL();
		console.log('getThreads', url);
	}

	loadState() {
		chrome.storage.local.get(['settings', 'thread_sort_mode'], (result) => {
			if (chrome.runtime.lastError) {
				console.error('Error loading state:', chrome.runtime.lastError);
				return;
			}
			this.state.settings = result.settings || {};
			this.state.thread_sort_mode = result.thread_sort_mode || 'date_desc'; // default to newest sort mode
			// Ensure all default settings are present if missing
			for (let key in this.settingsDefault) {
				if (!(key in this.state.settings)) {
					this.state.settings[key] = this.settingsDefault[key];
				}
			}
		});
	}

	saveState() {
		chrome.storage.local.set({
			settings: this.state.settings || {},
			thread_sort_mode: this.state.thread_sort_mode || 'date_desc',
		}, () => {
			if (chrome.runtime.lastError) {
				console.error('Error saving state:', chrome.runtime.lastError);
			}
		});
	}

	applyFontSizeSetting() {
		const font_size = this.getSetting('font_size');
		if (!font_size || typeof font_size != 'string' || font_size.length < 1) return;
		// validate that the font size ends with em and is a number from 0.5 to 1.5
		const font_size_num = parseFloat(font_size.replace('em', ''));
		if (isNaN(font_size_num) || font_size_num < 0.5 || font_size_num > 1.5 || !font_size.endsWith('em')) return;
		$('body').css({ fontSize: font_size });
	}

	getSetting(key) {
		if ('settings' in this.state && this.state.settings && typeof this.state.settings == 'object' && key in this.state.settings) {
			return this.state.settings[key];
		}
		return this.settingsDefault[key] || null;
	}

	updateSettings(newSettings) {
		let validSettings = {};
		let invalidParams = [];
		for (let key in newSettings) {
			if (this.settingsSchema[key]) {
				if (typeof newSettings[key] === this.settingsSchema[key]) {
					validSettings[key] = newSettings[key];
				} else if (this.settingsSchema[key] === 'number' && !isNaN(newSettings[key] * 1)) {
					validSettings[key] = newSettings[key] * 1;
				} else if (this.settingsSchema[key] === 'boolean' && ["true", "false"].indexOf(newSettings[key].toString().toLowerCase()) > -1) {
					validSettings[key] = newSettings[key].toString().toLowerCase() === 'true' ? true : false;
				} else {
					invalidParams.push(key);
				}
			} else {
				invalidParams.push(key);
			}
		}

		if (invalidParams.length > 0) {
			const invStr = invalidParams.join(', ');
			this.feed(`Invalid setting or type for parameter(s): ${invStr}`, true);
		} else {
			// merge partial settings with existing settings
			this.state.settings = { ...this.state.settings, ...validSettings };
			this.saveState();
			this.feed("Settings updated.")
		}
	}

	buildSettingsForm() {
		$('#reply-container').empty().append('<h2>Extension Settings</h2>');
		console.log('building settings form');

		// Create cancel button
		const cancelIcon = bsi('x-lg') || '❌';
		const cancelLink = $(`<a href="#" class="btn btn-sm btn-secondary float-end" id="exit_settings">${cancelIcon} Cancel</a>`);
		cancelLink.on('click', e => {
			e.preventDefault();
			$('#reply-container').empty();
			setTimeout(() => { this.getThreads }, 100);
		});
		$('#reply-container').append(cancelLink, '<br><br>');

		const settingsForm = $(`<form></form>`);

		console.log('building settings form body', this.settingsDefault);
		for (var key in this.settingsDefault) {
			console.log('building setting for key:', key, this.settingsDefault[key]);
			var input = null;
			var label = `<label for="${key}" title="${(desc ? desc : '')}">${key.replace(/_/g, ' ').toUpperCase()}</label>`;
			if (key == 'font_size') { // font size dropdown
				input = $(`<select name="font_size"></select>`);
				const options = ['0.5em', '0.6em', '0.7em', '0.8em', '0.9em', '1em', '1.1em', '1.2em', '1.3em', '1.4em', '1.5em'];
				const dflt = this.state.settings?.[key] || this.settingsDefault?.[key] || null;
				for (var j = 0; j < options.length; j++) {
					const opt = options[j];
					const selected = dflt == opt ? ' selected' : '';
					input.append(`<option value="${opt}"${selected}>${opt}</option>`);
				}
			} else {
				const typ = typeof this.settingsDefault[key] === 'number' ? 'number' : 'text';
				const val = this.state.settings?.[key] || this.settingsDefault[key];
				input = $(`<input type="${typ}" title="${(desc ? desc : '')}" name="${key}" value="${val}">`);
			}
			console.log('building setting for key:', key, input);
			if (!input) continue; // skip if no input
			if (label) settingsForm.append(label, '<br>');
			settingsForm.append(input, '<br>');
		}

		// append submit button
		settingsForm.append(`<br><br><input type="submit" value="Save Settings" class="btn-primary form-control"><br><br>`);
		settingsForm.on('submit', (e) => {
			e.preventDefault();
			for (var key in this.settingsDefault) {
				const input = settingsForm.find(`[name="${key}"]`);
				if (input.length < 1) continue;
				const val = input.val();
				if (Array.isArray(this.settingsDefault[key])) {
					const checkedBoxes = input.find('input[type="checkbox"]:checked');
					const checkedValues = [];
					checkedBoxes.each((i, el) => {
						checkedValues.push($(el).val());
					});
					this.updateSettings({ [key]: checkedValues });
				} else if (this.settingsSchema?.[key] == 'boolean') {
					this.updateSettings({ [key]: input.is(':checked') });
				} else {
					this.updateSettings({ [key]: val });
				}
			}
			this.saveState();
			this.updateConversionRates();
			this.applyFontSizeSetting();
			$('#exit_settings').trigger('click');
		});
		$('#reply-container').append(settingsForm);
	}
}
